subroutine init

! planar Supernova remnant colliding with spherical solar wind

!=======================================================================
! GLOBALS
use global
use zone
use solarwind

IMPLICIT NONE

! LOCALS
INTEGER :: i, j, k
REAL :: ridt, xvel, yvel, zvel, width, widthz, widthy
REAL :: xmin, xmax, ymin, ymax, zmin, zmax
REAL :: distcloud, xphys, yphys, theta, ramb, pamb

!--------------------------------------------------------------------------------
! Set up geometry and boundary conditions of grid
!
! Boundary condition flags : nleft, nright
!   = 0  :  reflecting boundary condition
!   = 1  :  inflow/outflow boundary condition (zero gradients)
!   = 2  :  fixed inflow boundary condition (values set by dinflo, pinflo, etc.)
!   = 3  :  periodic (nmax+1 = nmin; nmin-1 = nmax)
!
! Geometry flag : ngeom                         |  Cartesian:
!   = 0  :  planar                              |    gx = 0, gy = 0, gz = 0   (x,y,z)
!   = 1  :  cylindrical radial                  |  Cylindrical:
!   = 2  :  spherical   radial             3D= {     gx = 1, gy = 3, gz = 0   (s,phi,z)
!   = 3  :  cylindrical angle                   |
!   = 4  :  spherical polar angle (theta)       |  Spherical:
!   = 5  :  spherical azimu angle (phi)         |    gx = 2, gy = 4, gz = 5   (r,theta,phi)

! Define the computational grid...

ngeomx = 0 !cartesian = z
ngeomy = 1 !cylindrical = r
ngeomz = 0 !not used

nleftx = 1
nrightx= 1
nlefty = 0
nrighty= 1
nleftz = 0
nrightz= 0

xmin   = 0.0	! should be 0
xmax   = 12	! in AU
ymin   = 0.0	! should be 0
ymax   = 4	! in AU
zmin   = 0.0	! unused/2d problem
zmax   = 1.0	! unused/2d problem

! If any dimension is angular, multiply coordinates by pi...
if(ngeomy >= 3) then
   ymin = ymin * pi
   ymax = ymax * pi
endif
if(ngeomz >= 3) then
   zmin = zmin * pi
   zmax = zmax * pi
endif

!======================================================================

!units:
![M]=10^12 g
![T]=yr
![L]=AU

gam    = 5. / 3.
gamm = gam - 1.0

! ambient material
pamb = 3.27e4 		!pressure
ramb = 5.60e3 		!density

! solar wind properties
zinj = 4		! z-coordinate of injection disk
radinj = 0.25		! 1e18 ! injection disk radius
widthinj = 1.6		! width of injection disk (radinj/widthinj <= r <= radinj)
vinj = 88.75 		! velocity of injected material
pinj = pamb 		! pressure of injected material
rinj = 20*4.54e4 	! density of injected material

! SNR shock
ncycleshock = 4500 	!number of cycles to wait for solar wind to settle before shock
vshock = 128.6 		!velocity of shock material
pshock = 1.23e8 	!pressure of shock material
rshock = 2.24e4 	!density of shock material

!=======================================================================
! set time and cycle counters

time   = 0.0
timep  = 0.0
timem  = 0.0
ncycle = 0
ncycp  = 0
ncycd  = 0
ncycm  = 0
nfile  = 0

! Set up grid coordinates

call grid(imax,xmin,xmax,zxa,zxc,zdx)
call grid(jmax,ymin,ymax,zya,zyc,zdy)
call grid(kmax,zmin,zmax,zza,zzc,zdz)

if (ndim <= 2) zzc(1) = 0.0
if (ndim == 1) zyc(1) = 0.0

!======================================================================
! Log parameters of problem in history file

write (8,"('Collision between planar supernova remnant and radial solar wind in cylindrical coordinates.')")

write (8,"('Grid dimensions: ',i4,' x ',i4)") imax, jmax

write (8,*) 
write (8,"('Units: [T]=years, [M]=megatons (metric), [L]=astronomical units')")

write (8,*) 
write (8,"('Ratio of specific heats = ',f5.3)") gam
write (8,*) 
write (8,"('Supernova remnant:')")
write (8,"('  pressure=',e11.5)") pshock
write (8,"('  velocity=',e11.5)") vshock
write (8,"('  density=',e11.5)") rshock
write (8,*) 

write (8,"('Solar wind at injection radius:')")
write (8,"('  z-origin of injection disc=',f5.3)") zinj
write (8,"('  radius of injection disc=',f5.3)") radinj
write (8,"('  pressure=',e11.5)") pinj
write (8,"('  velocity=',e11.5)") vinj
write (8,"('  density=',e11.5)") rinj
write (8,*) 

write (8,"('Ambient medium / ISM:')")
write (8,"('  pressure=',e11.5)") pamb
write (8,"('  density=',e11.5)") ramb
write (8,*) 

! initialize grid:

do k = 1, kmax
 do j = 1, jmax
  do i = 1, imax
    xphys=zxc(i) !float(i)/(xmax-xmin)
    yphys=zyc(j) !float(j)/(ymax-ymin)
    
    distcloud=sqrt((xphys-zinj)**2+(yphys)**2) !distance to cloud/wind center/'sun'
    theta=atan2(yphys,xphys-zinj) !angle between z-axis and vector pointing from sun to current position
    
    !injection radius
    if(radinj/widthinj < distcloud .and. distcloud < radinj .and. j>0) then
      zro(i,j,k) = rinj
      zpr(i,j,k) = pinj
      zux(i,j,k) = vinj*cos(theta)
      zuy(i,j,k) = vinj*sin(theta)
    !fixed density inside injection radius
    elseif(distcloud <= radinj/widthinj) then
      zro(i,j,k) = rinj
      zpr(i,j,k) = pinj
      zux(i,j,k) = 0
      zuy(i,j,k) = 0
    !shock (disabled here because it should only start later)
!     elseif(i==5) then
!       zpr(i,j,k) = 15
!       zro(i,j,k) = 1
!       zux(i,j,k) = 0.
!       zuy(i,j,k) = 0.
    !ambient medium
    else
      zro(i,j,k) = ramb
      zpr(i,j,k) = pamb
      zux(i,j,k) = 0.
      zuy(i,j,k) = 0.
    endif
    zuz(i,j,k) = 0.
    zfl(i,j,k) = 0.
  enddo
 enddo
enddo

!########################################################################
! Compute Courant-limited timestep

ridt = 0.
if(ndim == 1) then
  do i = 1, imax
    svel = sqrt(gam*zpr(i,1,1)/zro(i,1,1))/zdx(i)
    xvel = abs(zux(i,1,1)) / zdx(i)
    ridt = max(xvel,svel,ridt)
  enddo
else if(ndim == 2) then
  do j = 1, jmax
   do i = 1, imax
     widthy = zdy(j)
     if(ngeomy > 2) widthy = widthy*zxc(i)
     width  = min(zdx(i),widthy)
     svel = sqrt(gam*zpr(i,j,1)/zro(i,j,1))/width
     xvel = abs(zux(i,j,1)) / zdx(i)
     yvel = abs(zuy(i,j,1)) / widthy
     ridt = max(xvel,yvel,svel,ridt)
   enddo
  enddo
else if(ndim == 3) then 
  do k = 1, kmax
   do j = 1, jmax
    do i = 1, imax
      widthy = zdy(j)
      widthz = zdz(k)
      if(ngeomy > 2) widthy = widthy*zxc(i)
      if(ngeomz > 2) widthz = widthz*zxc(i)
      if(ngeomz == 5) widthz = widthz*sin(zyc(j))
      width  = min(zdx(i),widthy,widthz)
      svel = sqrt(gam*zpr(i,j,k)/zro(i,j,k))/width
      xvel = abs(zux(i,j,k)) / zdx(i)
      yvel = abs(zuy(i,j,k)) / widthy
      zvel = abs(zuz(i,j,k)) / widthz
      ridt = max(xvel,yvel,zvel,svel,ridt)
    enddo
   enddo
  enddo
endif

dt = courant / ridt 

return
end

!#########################################################################

subroutine grid( nzones, xmin, xmax, xa, xc, dx )

! Create grid to cover physical size from xmin to xmax
! number of physical grid zones is nzones
!
! xa(1) is left boundary location - at xmin
! xa(nzones+1) is right boundary location - at xmax
!----------------------------------------------------------------------
IMPLICIT NONE

! LOCALS
integer :: nzones, n
real, dimension(nzones) :: xa, dx, xc
real :: dxfac, xmin, xmax

!=======================================================================

dxfac = (xmax - xmin) / float(nzones)
do n = 1, nzones
  xa(n) = xmin + (n-1)*dxfac
  dx(n) = dxfac
  xc(n) = xa(n) + 0.5*dx(n)
enddo

return
end
