Module UNEDF
  !=================================================================== 
  ! M.Kortelainen & M.Stoitsov, 2009-2010
  ! UNEDF interface for Skyrme, DME(minnesota pot.) and other DFT solvers 
  !=================================================================== 
  Implicit None
  !
  Character(16), Private :: Version='2'
  !
  ! Version History
  !
  ! ver#3 (Markus) CExPar added
  ! ver#2 (Markus) Added different options for Hartree part (HARTREEselect)
  ! 
  ! HARTREEselect = -1 : No Hartree at all
  ! HARTREEselect =  0 : Use exact Hartree
  ! HARTREEselect =  1 : Use NV or PSA DME Hartree (depending on DMEorder)
  ! HARTREEselect =  2 : Use Naive Taylor DME for Hartree
  !
  ! ver#1 (Markus) Basic coding and testing. HO trap included.
  !                INM properties not yet fully tested.
  !
  ! DMEorder = -1 : Skyrme functional
  ! DMEorder =  0 : DME of Minn. with Negele-Vauthering expansion
  ! DMEorder =  1 : DME of Minn. with Phase-space averaging
  !
  ! t for Uamplitudes(t,*)
  ! 0 -> 0,0
  ! 1 -> 1,1
  ! 2 -> 0,1
  ! 3 -> 1,0
  ! n for Uamplitudes(*,n)
  ! 0 -> U
  ! 1 -> dU/dRHO_0
  ! 2 -> dU/dRHO_1
  ! 3 -> d2U/(dRHO_0*dRHO_0)
  ! 4 -> d2U/(dRHO_1*dRHO_1) 
  ! 5 -> d2U/(dRHO_0*dRHO_1) 
  ! 6 -> dU/d(TAU_0)   
  ! 7 -> dU/d(Delta RHO_0)   
  !-------------------------------------------------------------------
  !
  ! === PUBLIC VARIABLES ===
  !
  ! Use pointers to prevent conflicts with UNEDF public variabes
  ! Example: Use UNEDF, pr=>my_pr, ipr=>my_ipr, Crho=>my_Crho ...
  !
  Integer, Parameter, Public :: pr=Kind(1.0D0), ipr=Kind(1)                   ! to set the precision of the DFT solver
  Logical, Public :: use_charge_density, use_cm_cor           
  Real(pr), Public, Dimension(0:3,0:7) :: Urhorho,Urhotau,UrhoDrho,Unablarho  ! ph DME amplitudes   
  Real(pr), Public, Dimension(0:3,0:7) :: UJnablarho,UrhonablaJ,UJJ
  Real(pr), Public, Dimension(0:3,0:7) :: Urhorhopr                           ! pp amplitudes   
  Real(pr), Public, Dimension(0:1) :: UEnonstdr,UFnonstdr,URnonstdr           ! Other amplitudes  
  Real(pr), Public :: hbzero,sigma,e2charg,CExPar                             ! hbr^2/2m, DD sigma, e^2 charge, coul. exchange parameter
  Real(pr), Public, Dimension(0:1) :: Crho,Cdrho,Ctau,CrDr,CrdJ,CJ,CpV0,CpV1  ! basic coupling constants
  Real(pr), Public :: E_NM,K_NM,SMASS_NM,RHO_NM,ASS_NM,LASS_NM,VMASS_NM,P_NM,KA_NM
  Real(pr), Public, Dimension(0:1) :: CHrho                                   ! Crho(0,1) from the Hartree term in NM
  Logical, Public :: use_DME3N_terms,use_j2terms
  Integer(ipr), Public :: DMEorder,DMElda,HARTREEselect
  !  
  ! === PRIVATE VARIABLES ===
  !
  Real(pr), Private, Dimension(0:1) :: nuCrho,nuCdrho,nuCtau,nuCrDr           ! basic coupling constants in natural units
  Real(pr), Private, Dimension(0:1) :: nuCrdJ,nuCJ,nuCpV0,nuCpV1              !
  Real(pr), Private :: t0,t1,t2,t3,x0,x1,x2,x3,b4,b4p,te,to
  Real(pr), Private :: nuLambda,nufpi                                         ! parameters associated to natural units
  Real(pr), Private, Dimension(0:1) :: Cnrho,CJdr                             ! hidden and always zero
  Integer(ipr), Private :: i_cut                                              ! dmeorder: -1=Standard Skyrme, 0=LO, 1=NLO, 2=N2LO  
  Real(pr), Private :: Pi,eps                                                 ! dmelda: 0=Kf-LDA, 1=CB-LDA
  Real(pr), Private :: kfconst,CK                                             ! (3Pi^2/2)^(1/3)
  Real(pr), Parameter, Private :: mevfm=197.30_pr; 
  Real(pr), Private :: rho(0:1),tau(0:1),nrho2(0:1),lrho(0:1)
  Real(pr), Private, Dimension(0:1) :: arhorho,arhodrho,arhotau,ajj
  Real(pr), Private, Dimension(0:1) :: darhorho,darhodrho,darhotau,dajj
  Real(pr), Private, Dimension(0:1) :: ddarhorho,ddarhodrho,ddarhotau,ddajj
  Real(pr), Private, Dimension(0:1) :: anrnr,danrnr,ddanrnr
  Real(pr), Private :: u2,u3,u4,u5,u6,u7,u8,u9,u10,u11,u12,mpi2,mpi3,mpi4     !  optimization
  Real(pr), Private :: mpi
  Real(pr), Private, Dimension(1:3) :: mimu,mivi,miwi,mimi,mibi,mihi
  Real(pr), Private :: ac2,ac3,acoord                                                                   !! not used at the moment
  Parameter(acoord=0.50_pr,ac2=4.0_pr*(acoord**2-acoord+0.50_pr),ac3=2.0_pr*(acoord**2-acoord+0.50_pr)) !! not used at the moment
  Character (30) :: FunctionalName 
  Logical, Private :: Print_Namelist
  Real(pr), Private :: V0trap,hwtrap,Vcutoff
  Parameter(Vcutoff=0.0_pr)
  !
Contains
  !
  !=======================================================================================================
  Subroutine calculate_U_parameters(rho0_in,rho1_in,tau0_in,tau1_in,laprho0,laprho1,nablarho0s,nablarho1s)
    !-----------------------------------------------------------------------------------------------------
    Implicit None
    Real(pr), Intent(in) :: rho0_in,rho1_in,tau0_in,tau1_in
    Real(pr), Intent(in), Optional :: &
         nablarho0s,nablarho1s,laprho0,laprho1
    Integer(ipr) :: t,i,j,k,l
    Real(pr) :: u,du,ddu,dtu,dlu
    Real(pr), Dimension(1:3) :: frhorho,frhotau,fjj
    Real(pr), Dimension(1:3) :: dfrhorho,dfrhotau,dfjj
    Real(pr), Dimension(1:3) :: ddfrhorho,ddfrhotau,ddfjj
    Real(pr), Dimension(1:3) :: hrhorho,hrhoDrho,dhrhoDrho,ddhrhoDrho
    Real(pr), Dimension(1:3) :: VC,WC,VS,VCtilde
    Real(pr) :: y,dy,ddy,marc,dmarc,ddmarc,mlog,dmlog,ddmlog
    Real(pr) :: ucut
    !
    ucut=0.4_pr; 
    !    
    rho(0)=rho0_in; rho(1)=rho1_in; 
    tau(0)=tau0_in; tau(1)=tau1_in; 
    !
    lrho=0.0_pr; nrho2=0.0_pr; 
    If (Present(laprho0)) lrho(0)=laprho0 
    If (Present(laprho1)) lrho(1)=laprho1
    If (Present(nablarho0s)) nrho2(0)=nablarho0s 
    If (Present(nablarho1s)) nrho2(1)=nablarho1s
    !
    arhorho = 0.0_pr ; arhodrho = 0.0_pr ; arhotau = 0.0_pr ; ajj = 0.0_pr ;
    darhorho = 0.0_pr ; darhodrho = 0.0_pr ; darhotau = 0.0_pr ; dajj = 0.0_pr ;
    ddarhorho = 0.0_pr ; ddarhodrho = 0.0_pr ; ddarhotau = 0.0_pr ; ddajj = 0.0_pr ;
    anrnr = 0.0_pr ; danrnr = 0.0_pr ; ddanrnr = 0.0_pr ;
    !
    VC = (miVi*(miWi + (miBi - miHi)/2.0_pr))
    WC = (-miVi*miHi/2.0_pr)
    VS = (miVi*miBi/2.0_pr)
    VCtilde = (miVi*miMi)
    !
    ! U and partial derivatives with respect of rho_0 in Thomas Fermi approximation
    !
    u=0.0_pr; du=0.0_pr; ddu=0.0_pr; dtu=0.0_pr; dlu=0.0_pr
    If (dmeorder.Ge.0) Then
       If (dmelda.Eq.0) Then
          ! density dependent LDA
          u=(kfconst/mpi)*rho(0)**(1.0_pr/3.0_pr)
          du=(1.0_pr/3.0_pr)*u/(rho(0)+eps)                 ! u'(RHO_0)   
          ddu=-(2.0_pr/9.0_pr)*u/(rho(0)**2+eps)            ! u''(RHO_0)  
          dtu=0.0_pr                                        ! u'(TAU_0)   
          dlu=0.0_pr                                        ! u'(DeltaRHO_0) 
       Else 
          ! density and gradient dependent LDA       
          u=Sqrt(Abs((5.0_pr/3.0_pr)*(tau(0)-0.250_pr*lrho(0))/(rho(0)+eps)))/mpi 
          du=-0.50_pr*u/(rho(0)+eps)                        ! u'(RHO_0)   
          ddu=0.750_pr*u/(rho(0)**2+eps)                    ! u''(RHO_0)
          dtu=0.50_pr*u/(Abs(tau(0)-0.250_pr*lrho(0))+eps)  ! u'(TAU_0)   
          dlu=-0.250_pr*dtu                                 ! u'(DeltaRHO_0) 
       Endif
    Endif
    !
    ! Partial optimiztion
    u2=u*u; u3=u2*u; u4=u3*u; u5=u4*u; u6=u5*u; u7=u6*u; u8=u7*u; u9=u8*u; u10=u9*u; u11=u10*u; u12=u11*u;    
    !
    ! Expressions for the Fock part
    !
    select case(dmeorder)
    case(-1)
      frhorho = 0.0_pr ; frhotau = 0.0_pr ; fjj = 0.0_pr
      dfrhorho = 0.0_pr ; dfrhotau = 0.0_pr ; dfjj = 0.0_pr
      ddfrhorho = 0.0_pr ; ddfrhotau = 0.0_pr ; ddfjj = 0.0_pr
    case(0) !! NV description
      if (u.gt.ucut) then 
        frhorho = (3.0_pr*Sqrt(1.0_pr/mimu)*Sqrt(mimu)*Pi**1.5_pr*(2.0_pr*Sqrt(mimu)*((84.0_pr*mimu**2 &
           - 44.0_pr*mpi**2*mimu*u2 + mpi**4*u4)/Exp((mpi**2*u2)/mimu) + 2.0_pr*(-42.0_pr*mimu**2 &
           + 64.0_pr*mpi**2*mimu*u2 + 9.0_pr*mpi**4*u4)) + mpi**3*Sqrt(Pi)*u3*(-105.0_pr*mimu &
           + 2.0_pr*mpi**2*u2)*Erf((mpi*u)/Sqrt(mimu))))/(8.0_pr*mpi**8*u**8)

        frhotau = (105.0_pr*Pi**1.5_pr*(8.0_pr*mimu**2 - 12.0_pr*mpi**2*mimu*u2 - 2.0_pr*mpi**4*u4 &
           + (4.0_pr*mimu*(-2.0_pr*mimu + mpi**2*u2))/Exp((mpi**2*u2)/mimu) + 5.0_pr*mpi**3*Sqrt(mimu) &
           *Sqrt(Pi)*u3*Erf((mpi*u)/Sqrt(mimu))))/(8.0_pr*mpi**10*Sqrt(1.0_pr/mimu)*u**10)

        fjj = ((1/mimu)**2.5_pr*Pi**1.5_pr*(-mimu + (2.0_pr*(mimu - 2.0_pr*mpi**2*u2))/Exp((mpi**2*u2)/mimu))) &
           /(192.0_pr*mpi**2*u2)

        dfrhorho = (9.0_pr*Sqrt(1.0_pr/mimu)*Sqrt(mimu)*Pi**1.5*(2*Sqrt(mimu)*(-224.0_pr*mimu**2 &
           + 32.0_pr*mpi**2*mimu*u2 - 7.0_pr*mpi**4*u4 + 8.0_pr*Exp((mpi**2*u2)/mimu)*(28.0_pr*mimu**2 &
           - 32.0_pr*mpi**2*mimu*u2 - 3.0_pr*mpi**4*u4))- Exp((mpi**2*u2)/mimu)*mpi**3*Sqrt(Pi)*u3 &
           *(-175.0_pr*mimu + 2.0_pr*mpi**2*u2)*Erf((mpi*u)/Sqrt(mimu))))/(8.0_pr*Exp((mpi**2*u2)/mimu) &
           *mpi**8*u**9)

        dfrhotau = (105.0_pr*Pi**1.5_pr*(2.0_pr*(40.0_pr*mimu**2 - 8.0_pr*mpi**2*mimu*u2 + mpi**4*u4) &
           + Exp((mpi**2*u2)/mimu)*(-80.0_pr*mimu**2 + 96.0_pr*mpi**2*mimu*u2 + 12.0_pr*mpi**4*u4 &
           - 35.0_pr*mpi**3*Sqrt(mimu)*Sqrt(Pi)*u3*Erf((mpi*u)/Sqrt(mimu)))))/(8.0_pr*Exp((mpi**2*u2)/mimu) &
           *mpi**10*Sqrt(1.0_pr/mimu)*u**11)

        dfjj = ((1.0_pr/mimu)**3.5*Pi**1.5_pr*((-2.0_pr + Exp((mpi**2*u2)/mimu))*mimu**2 - 2.0_pr*mpi**2*mimu*u2 &
           + 4.0_pr*mpi**4*u4))/(96.0_pr*Exp((mpi**2*u2)/mimu)*mpi**2*u3)

        ddfrhorho = (-9.0_pr*Sqrt(1.0_pr/mimu)*Pi**1.5_pr*(8.0_pr*Exp((mpi**2*u2)/mimu)*mimu*(252.0_pr*mimu**2 &
           - 224.0_pr*mpi**2*mimu*u2 - 15.0_pr*mpi**4*u4) - 2.0_pr*(1008.0_pr*mimu**3 + 112.0_pr*mpi**2*mimu**2*u2 &
           + 73.0_pr*mpi**4*mimu*u4 + 6.0_pr*mpi**6*u6) - Exp((mpi**2*u2)/mimu)*mpi**3*Sqrt(mimu)*Sqrt(Pi)*u3 &
           *(-525.0_pr*mimu + 4.0_pr*mpi**2*u2)*Erf((mpi*u)/Sqrt(mimu))))/(4.0_pr*Exp((mpi**2*u2)/mimu)*mpi**8*u**10)

        ddfrhotau = (105.0_pr*Sqrt(1.0_pr/mimu)*Pi**1.5_pr*(-220.0_pr*mimu**3 - 4.0_pr*mpi**2*mimu**2*u2 &
           - 13.0_pr*mpi**4*mimu*u4 - mpi**6*u6 + Exp((mpi**2*u2)/mimu)*mimu*(220.0_pr*mimu**2 &
           - 216.0_pr*mpi**2*mimu*u2 - 21.0_pr*mpi**4*u4 + 70.0_pr*mpi**3*Sqrt(mimu)*Sqrt(Pi)*u3*Erf((mpi*u) &
           /Sqrt(mimu)))))/(2.0_pr*Exp((mpi**2*u2)/mimu)*mpi**10*u**12)

        ddfjj = ((1.0_pr/mimu)**4.5_pr*Pi**1.5_pr*(-3.0_pr*(-2.0_pr + Exp((mpi**2*u2)/mimu))*mimu**3 &
           + 6.0_pr*mpi**2*mimu**2*u2 + 8.0_pr*mpi**4*mimu*u4 - 8.0_pr*mpi**6*u6))/(96.0_pr*Exp((mpi**2*u2) &
           /mimu)*mpi**2*u4)
      else
        frhorho = ((1.0_pr/mimu)**4.5_pr*Pi**1.5_pr*(6930.0_pr*mimu**3 - 363.0_pr*mpi**4*mimu*u4 &
           + 112.0_pr*mpi**6*u6))/55440.0_pr

        frhotau = ((1.0_pr/mimu)**5.5_pr*Pi**1.5_pr*(-10296.0_pr*mimu**3 + 4004.0_pr*mpi**2*mimu**2*u2 &
           - 936.0_pr*mpi**4*mimu*u4 + 165.0_pr*mpi**6*u6))/164736.0_pr

        fjj = ((1.0_pr/mimu)**5.5_pr*Pi**1.5_pr*(12.0_pr*mimu**4 - 72.0_pr*mpi**2*mimu**3*u2 &
           + 60.0_pr*mpi**4*mimu**2*u4 - 28.0_pr*mpi**6*mimu*u6 + 9.0_pr*mpi**8*u**8))/(2304.0_pr*mpi**2*u2)

        dfrhorho = (mpi**4*(1.0_pr/mimu)**4.5_pr*Pi**1.5_pr*u3*(-121.0_pr*mimu + 56.0_pr*mpi**2*u2))/4620.0_pr

        dfrhotau = (mpi**2*(1.0_pr/mimu)**5.5_pr*Pi**1.5_pr*u*(4004.0_pr*mimu**2 - 1872.0_pr*mpi**2*mimu*u2 &
           + 495.0_pr*mpi**4*u4))/82368.0_pr

        dfjj = ((1.0_pr/mimu)**5.5_pr*Pi**1.5_pr*(-12.0_pr*mimu**4 + 60.0_pr*mpi**4*mimu**2*u4 &
           - 56.0_pr*mpi**6*mimu*u6 + 27.0_pr*mpi**8*u**8))/(1152.0_pr*mpi**2*u3)

        ddfrhorho = -(mpi**4*(1.0_pr/mimu)**5.5_pr*Pi**1.5_pr*u2*(18876.0_pr*mimu**2 &
           - 14560.0_pr*mpi**2*mimu*u2 + 5355.0_pr*mpi**4*u4))/240240.0_pr

        ddfrhotau = (mpi**2*(1.0_pr/mimu)**6.5_pr*Pi**1.5_pr*(12012.0_pr*mimu**3 - 16848.0_pr*mpi**2*mimu**2*u2 &
           + 7425.0_pr*mpi**4*mimu*u4 - 2002.0_pr*mpi**6*u6))/247104.0_pr

        ddfjj = ((1.0_pr/mimu)**6.5_pr*Pi**1.5_pr*(180.0_pr*mimu**5 + 300.0_pr*mpi**4*mimu**3*u4 &
           - 840.0_pr*mpi**6*mimu**2*u6 + 675.0_pr*mpi**8*mimu*u**8 - 308.0_pr*mpi**10*u**10)) &
           /(5760.0_pr*mpi**2*u4)
      end if
    case(1) !! PSA description
      if (u.gt.ucut) then 
        frhorho = (3.0_pr*Sqrt(1.0_pr/mimu)*Pi**1.5_pr*(20.0_pr*mimu**2 - 36.0_pr*mpi**2*mimu*u2 &
           + 3.0_pr*mpi**4*u4 + (-20.0_pr*mimu**2 + 16.0_pr*mpi**2*mimu*u2 + 3.0_pr*mpi**4*u4)  &
           /Exp((mpi**2*u2)/mimu) + 10.0_pr*mpi**3*Sqrt(mimu)*Sqrt(Pi)*u3*Erf((mpi*u)/Sqrt(mimu)))) &
           /(40.0_pr*mpi**6*u6)

        frhotau = (3.0_pr*Sqrt(1.0_pr/mimu)*Pi**1.5_pr*(2.0_pr*mimu - mpi**2*u2 - (2.0_pr*mimu  &
           + mpi**2*u2)/Exp((mpi**2*u2)/mimu)))/(8.0_pr*mpi**6*u6)

        fjj = (3.0_pr*Sqrt(1.0_pr/mimu)*Pi**1.5_pr*(2.0_pr*mimu - mpi**2*u2 - (2.0_pr*mimu &
           + mpi**2*u2)/Exp((mpi**2*u2)/mimu)))/(16.0_pr*mpi**6*u6)

        dfrhorho = (-9.0_pr*(1.0_pr/mimu)**1.5_pr*Pi**1.5_pr*(-20.0_pr*mimu**3 + 4.0_pr*mpi**2*mimu**2*u2 &
           + 3.0_pr*mpi**4*mimu*u4 + mpi**6*u6 + Exp((mpi**2*u2)/mimu)*mimu*(20.0_pr*mimu**2 &
           - 24.0_pr*mpi**2*mimu*u2 + mpi**4*u4 + 5.0_pr*mpi**3*Sqrt(mimu)*Sqrt(Pi)*u3*Erf((mpi*u) &
           /Sqrt(mimu)))))/(20.0_pr*Exp((mpi**2*u2)/mimu)*mpi**6*u**7)

        dfrhotau = (-3.0_pr*(1.0_pr/mimu)**1.5_pr*Pi**1.5_pr*(-6.0_pr*mimu**2 - 4.0_pr*mpi**2*mimu*u2 &
          - mpi**4*u4 + 2.0_pr*Exp((mpi**2*u2)/mimu)*mimu*(3.0_pr*mimu - mpi**2*u2)))/(4.0_pr*Exp((mpi**2*u2) &
          /mimu)*mpi**6*u**7)

        dfjj = (-3.0_pr*(1.0_pr/mimu)**1.5_pr*Pi**1.5_pr*(-6.0_pr*mimu**2 - 4.0_pr*mpi**2*mimu*u2 - mpi**4*u4 &
          + 2.0_pr*Exp((mpi**2*u2)/mimu)*mimu*(3.0_pr*mimu - mpi**2*u2)))/(8.0_pr*Exp((mpi**2*u2)/mimu) &
          *mpi**6*u**7)

        ddfrhorho = (9.0_pr*(1.0_pr/mimu)**2.5_pr*Pi**1.5_pr*(-140.0_pr*mimu**4 - 20.0_pr*mpi**2*mimu**3*u2 &
          + 7.0_pr*mpi**4*mimu**2*u4 + 7.0_pr*mpi**6*mimu*u6 + 2.0_pr*mpi**8*u**8 + Exp((mpi**2*u2)/mimu) &
          *mimu**2*(140.0_pr*mimu**2 - 120.0_pr*mpi**2*mimu*u2 + 3.0_pr*mpi**4*u4 + 20.0_pr*mpi**3*Sqrt(mimu) &
          *Sqrt(Pi)*u3*Erf((mpi*u)/Sqrt(mimu)))))/(20.0_pr*Exp((mpi**2*u2)/mimu)*mpi**6*u**8)

        ddfrhotau = (3.0_pr*(1.0_pr/mimu)**2.5_pr*Pi**1.5_pr*(42.0_pr*(-1.0_pr + Exp((mpi**2*u2)/mimu))*mimu**3 &
          - 2.0_pr*(16.0_pr + 5.0_pr*Exp((mpi**2*u2)/mimu))*mpi**2*mimu**2*u2 - 11.0_pr*mpi**4*mimu*u4 &
          - 2.0_pr*mpi**6*u6))/(4.0_pr*Exp((mpi**2*u2)/mimu)*mpi**6*u**8)

        ddfjj = (3*(1.0_pr/mimu)**2.5_pr*Pi**1.5_pr*(42.0_pr*(-1.0_pr + Exp((mpi**2*u2)/mimu))*mimu**3 &
          - 2.0_pr*(16.0_pr + 5.0_pr*Exp((mpi**2*u2)/mimu))*mpi**2*mimu**2*u2 - 11.0_pr*mpi**4*mimu*u4 &
          - 2.0_pr*mpi**6*u6))/(8.0_pr*Exp((mpi**2*u2)/mimu)*mpi**6*u**8)
      else
        frhorho = ((1.0_pr/mimu)**4.5_pr*Pi**1.5_pr*(12600.0_pr*mimu**3 - 1080.0_pr*mpi**4*mimu*u4 &
          + 427.0_pr*mpi**6*u6))/100800.0_pr

        frhotau = ((1.0_pr/mimu)**5.5_pr*Pi**1.5_pr*(-60.0_pr*mimu**3 + 30.0_pr*mpi**2*mimu**2*u2 &
          - 9.0_pr*mpi**4*mimu*u4 + 2.0_pr*mpi**6*u6))/960.0_pr

        fjj = ((1.0_pr/mimu)**5.5_pr*Pi**1.5_pr*(-60.0_pr*mimu**3 + 30.0_pr*mpi**2*mimu**2*u2 &
          - 9.0_pr*mpi**4*mimu*u4 + 2.0_pr*mpi**6*u6))/1920.0_pr

        dfrhorho = (mpi**4*(1.0_pr/mimu)**4.5_pr*Pi**1.5_pr*u3*(-720.0_pr*mimu + 427.0_pr*mpi**2*u2))/16800.0_pr

        dfrhotau = (mpi**2*(1.0_pr/mimu)**5.5_pr*Pi**1.5_pr*u*(5.0_pr*mimu**2 - 3.0_pr*mpi**2*mimu*u2 &
          + mpi**4*u4))/80.0_pr

        dfjj = (mpi**2*(1.0_pr/mimu)**5.5_pr*Pi**1.5_pr*u*(5.0_pr*mimu**2 - 3.0_pr*mpi**2*mimu*u2 &
          + mpi**4*u4))/160.0_pr

        ddfrhorho = -(mpi**4*(1.0_pr/mimu)**5.5_pr*Pi**1.5_pr*u2*(23760.0_pr*mimu**2 &
          - 23485.0_pr*mpi**2*mimu*u2 + 10836.0_pr*mpi**4*u4))/184800.0_pr

        ddfrhotau = (mpi**2*(1.0_pr/mimu)**6.5_pr*Pi**1.5_pr*(15.0_pr*mimu**3 - 27.0_pr*mpi**2*mimu**2*u2 &
          + 15.0_pr*mpi**4*mimu*u4 - 5.0_pr*mpi**6*u6))/240.0_pr

        ddfjj = (mpi**2*(1.0_pr/mimu)**6.5_pr*Pi**1.5_pr*(15.0_pr*mimu**3 - 27.0_pr*mpi**2*mimu**2*u2 &
          + 15.0_pr*mpi**4*mimu*u4 - 5.0_pr*mpi**6*u6))/480.0_pr
      end if
    case default
      stop "unknown dme order?"
    end select

    ! Expressions for the Hartree part
    hrhorho = 0.0_pr ; hrhoDrho = 0.0_pr ; dhrhoDrho = 0.0_pr ; ddhrhoDrho = 0.0_pr ;
    if (DMEorder.ge.0) then
    select case(HARTREEselect)
    case (-1,0) ! no DME hartree
      hrhorho = 0.0_pr ; hrhoDrho = 0.0_pr ; dhrhoDrho = 0.0_pr ; ddhrhoDrho = 0.0_pr ; ! DME Hartree off

    case (1) ! NV or PSA hartree
      hrhorho = ((1.0_pr/mimu)**1.5_pr*Pi**1.5)/8.0_pr !! general for all cases
      select case(dmeorder)
      case(-1) 
        hrhorho = 0.0_pr ; hrhoDrho = 0.0_pr ; dhrhoDrho = 0.0_pr ; ddhrhoDrho = 0.0_pr ; ! DME Hartree off
      case(0) ! NV
        if (u.gt.ucut) then 
          hrhoDrho = (-35.0_pr*Sqrt(1.0_pr/mimu)*Pi**1.5_pr*((-2.0_pr*mpi*u*(15.0_pr*mimu     &
            + mpi**2*u2))/Exp((mpi**2*u2)/(4.0_pr*mimu)) + 3.0_pr*Sqrt(mimu)*Sqrt(Pi)         &
            *(10.0_pr*mimu - mpi**2*u2)*Erf((mpi*u)/(2.0_pr*Sqrt(mimu)))))/(32.0_pr*mpi**7*u7)
          dhrhoDrho = (35.0_pr*(1.0_pr/mimu)**1.5_pr*Pi**1.5_pr*(-(mpi*u*(210.0_pr*mimu**2    &
            + 20.0_pr*mpi**2*mimu*u2 + mpi**4*u4)) + 15.0_pr*Exp((mpi**2*u2)/(4.0_pr*mimu))   &
            *mimu**1.5_pr*Sqrt(Pi)*(14.0_pr*mimu - mpi**2*u2)*Erf((mpi*u)/(2.0_pr*Sqrt(mimu))))) &
            /(32.0_pr*Exp((mpi**2*u2)/(4.0_pr*mimu))*mpi**7*u8)
          ddhrhoDrho = (-35.0_pr*(1.0_pr/mimu)**2.5_pr*Pi**1.5_pr*(-(mpi*u*(3360.0_pr*mimu**3 &
            + 380.0_pr*mpi**2*mimu**2*u2 + 26.0_pr*mpi**4*mimu*u4 + mpi**6*u6)) + 60.0_pr     &
            *Exp((mpi**2*u2)/(4.0_pr*mimu))*mimu**2.5_pr*Sqrt(Pi)*(56.0_pr*mimu               &
            - 3.0_pr*mpi**2*u2)*Erf((mpi*u)/(2.0_pr*Sqrt(mimu)))))/(64.0_pr*Exp((mpi**2*u2)   &
            /(4.0_pr*mimu))*mpi**7*u9)
        else
          hrhoDrho = ((1.0_pr/mimu)**5.5_pr*Pi**1.5_pr*(164736.0_pr*mimu**3                   &
            - 22880.0_pr*mpi**2*mimu**2*u2 + 1820.0_pr*mpi**4*mimu*u4 - 105.0_pr*mpi**6*u6))  &
            /1.0543104e7_pr
          dhrhoDrho = (-5.0_pr*mpi**2*(1.0_pr/mimu)**5.5_pr*Pi**1.5_pr*u*(4576.0_pr*mimu**2   &
            - 728.0_pr*mpi**2*mimu*u2 + 63.0_pr*mpi**4*u4))/5.271552e6_pr
          ddhrhoDrho = -(mpi**2*(1.0_pr/mimu)**6.5_pr*Pi**1.5_pr*(91520.0_pr*mimu**3          &
            - 43680.0_pr*mpi**2*mimu**2*u2 + 6300.0_pr*mpi**4*mimu*u4 - 539.0_pr*mpi**6*u6))  &
            /2.1086208e7_pr
        end if
      case(1) ! PSA
        hrhoDrho = ((1.0_pr/mimu)**2.5_pr*Pi**1.5_pr)/(64.0_pr*Exp((mpi**2*u2)/(4.0_pr*mimu)))
        dhrhoDrho = -(mpi**2*(1.0_pr/mimu)**3.5_pr*Pi**1.5_pr*u)/(128.0_pr*Exp((mpi**2*u2) &
           /(4.0_pr*mimu)))
        ddhrhoDrho = (mpi**2*(1.0_pr/mimu)**4.5_pr*Pi**1.5_pr*(-2.0_pr*mimu + mpi**2*u2))  &
           /(256.0_pr*Exp((mpi**2*u2)/(4.0_pr*mimu)))
      case default
        stop 'this should not happen...'
      end select

    case (2) ! Taylor DME for Hartree
      hrhorho = ((1.0_pr/mimu)**1.5_pr*Pi**1.5)/8.0_pr !! general for all cases
      hrhoDrho = Pi**1.5_pr/(64.0_pr*mimu**2.5_pr)
    case default
      stop "unknown Hartree selection?"
    end select  !!case(HARTREEselect)
    end if
    !
    arhorho = 0.0_pr ; arhotau = 0.0_pr ; arhoDrho = 0.0_pr ; ajj = 0.0_pr
    darhorho = 0.0_pr ; darhotau = 0.0_pr ; darhoDrho = 0.0_pr ; dajj = 0.0_pr
    ddarhorho = 0.0_pr ; ddarhotau = 0.0_pr ; ddarhoDrho = 0.0_pr ; ddajj = 0.0_pr

    ! Fock contributions
    Do i=1,3
      arhorho(0) = arhorho(0) -frhorho(i)*(VC(i)-4.0_pr*VCtilde(i) +3.0_pr*(VS(i) + WC(i)))
      darhorho(0) = darhorho(0) -dfrhorho(i)*(VC(i)-4.0_pr*VCtilde(i) +3.0_pr*(VS(i) + WC(i)))
      ddarhorho(0) = ddarhorho(0) -ddfrhorho(i)*(VC(i)-4.0_pr*VCtilde(i) +3.0_pr*(VS(i) + WC(i)))
      arhorho(1) = arhorho(1) -frhorho(i)*(VC(i) +3.0_pr*VS(i) - WC(i))
      darhorho(1) = darhorho(1) -dfrhorho(i)*(VC(i) +3.0_pr*VS(i) - WC(i))
      ddarhorho(1) = ddarhorho(1) -ddfrhorho(i)*(VC(i) +3.0_pr*VS(i) - WC(i))

      arhotau(0) = arhotau(0) -frhotau(i)*(VC(i)-4.0_pr*VCtilde(i) +3.0_pr*(VS(i) + WC(i)))
      darhotau(0) = darhotau(0) -dfrhotau(i)*(VC(i)-4.0_pr*VCtilde(i) +3.0_pr*(VS(i) + WC(i)))
      ddarhotau(0) = ddarhotau(0) -ddfrhotau(i)*(VC(i)-4.0_pr*VCtilde(i) +3.0_pr*(VS(i) + WC(i)))
      arhotau(1) = arhotau(1) -frhotau(i)*(VC(i) +3.0_pr*VS(i) - WC(i))
      darhotau(1) = darhotau(1) -dfrhotau(i)*(VC(i) +3.0_pr*VS(i) - WC(i))
      ddarhotau(1) = ddarhotau(1) -ddfrhotau(i)*(VC(i) +3.0_pr*VS(i) - WC(i))

      ajj(0) = ajj(0) +fjj(i)*(VC(i) - VS(i) + 3.0_pr*WC(i))
      dajj(0) = dajj(0) +dfjj(i)*(VC(i) - VS(i) + 3.0_pr*WC(i))
      ddajj(0) = ddajj(0) +ddfjj(i)*(VC(i) - VS(i) + 3.0_pr*WC(i))
      ajj(1) = ajj(1) +fjj(i)*(VC(i) - VS(i) - WC(i))
      dajj(1) = dajj(1) +dfjj(i)*(VC(i) - VS(i) - WC(i))
      ddajj(1) = ddajj(1) +ddfjj(i)*(VC(i) - VS(i) - WC(i))
    End do
    !
    arhoDrho = -arhotau/4.0_pr ; darhoDrho = -darhotau/4.0_pr ; ddarhoDrho = -ddarhotau/4.0_pr
    !
    ! Hartree contributions
    Do i=1,3
      arhorho(0) = arhorho(0) + hrhorho(i)*(4.0_pr*VC(i)-VCtilde(i))
      arhorho(1) = arhorho(1) + hrhorho(i)*(4.0_pr*WC(i)-VCtilde(i))

      arhoDrho(0) = arhoDrho(0) + hrhoDrho(i)*(4.0_pr*VC(i)-VCtilde(i))
      arhoDrho(1) = arhoDrho(1) + hrhoDrho(i)*(4.0_pr*WC(i)-VCtilde(i))
      darhoDrho(0) = darhoDrho(0) + dhrhoDrho(i)*(4.0_pr*VC(i)-VCtilde(i))
      darhoDrho(1) = darhoDrho(1) + dhrhoDrho(i)*(4.0_pr*WC(i)-VCtilde(i))
      ddarhoDrho(0) = ddarhoDrho(0) + ddhrhoDrho(i)*(4.0_pr*VC(i)-VCtilde(i))
      ddarhoDrho(1) = ddarhoDrho(1) + ddhrhoDrho(i)*(4.0_pr*WC(i)-VCtilde(i))

      anrnr(0) =  anrnr(0) - hrhoDrho(i)*(4.0_pr*VC(i)-VCtilde(i))
      anrnr(1) =  anrnr(1) - hrhoDrho(i)*(4.0_pr*WC(i)-VCtilde(i))
      danrnr(0) =  danrnr(0) - dhrhoDrho(i)*(4.0_pr*VC(i)-VCtilde(i))
      danrnr(1) =  danrnr(1) - dhrhoDrho(i)*(4.0_pr*WC(i)-VCtilde(i))
      ddanrnr(0) =  ddanrnr(0) - ddhrhoDrho(i)*(4.0_pr*VC(i)-VCtilde(i))
      ddanrnr(1) =  ddanrnr(1) - ddhrhoDrho(i)*(4.0_pr*WC(i)-VCtilde(i))
    End do
    !
    Urhorho=0.0_pr   ; Urhotau=0.0_pr
    UrhoDrho=0.0_pr  ; Unablarho=0.0_pr
    UJnablarho=0.0_pr; UrhonablaJ=0.0_pr
    Urhorhopr=0.0_pr ; UJJ=0.0_pr
    !
    ! Notations for Uamplitudes(0:3,0:7)
    ! t for Uamplitudes(t,*)
    ! 0 -> 0,0
    ! 1 -> 1,1
    ! 2 -> 0,1
    ! 3 -> 1,0
    ! n for Uamplitudes(*,n)
    ! 0 -> U
    ! 1 -> dU/dRHO_0
    ! 2 -> dU/dRHO_1
    ! 3 -> d2U/(dRHO_0*dRHO_0)
    ! 4 -> d2U/(dRHO_1*dRHO_1) 
    ! 5 -> d2U/(dRHO_0*dRHO_1) 
    ! 6 -> dU/d(TAU_0)   
    ! 7 -> dU/d(Delta RHO_0)   
    !
    !! 2N terms
    Do t=0,1
       Urhorho(t,0)=Crho(t)+Cdrho(t)*rho(0)**sigma +arhorho(t)
       Urhotau(t,0)=Ctau(t)+arhotau(t)
       UrhoDrho(t,0)=Crdr(t)+arhoDrho(t)
       UJJ(t,0)=CJ(t)+ajj(t)
       Unablarho(t,0)=anrnr(t)
       UrhonablaJ(t,0)=Crdj(t)
       UJnablarho(t,0)=Cjdr(t)

       Urhorho(t,1)=sigma*Cdrho(t)*(rho(0)**sigma)/(rho(0)+eps) &
            +darhorho(t)*du
       Urhotau(t,1)=darhotau(t)*du
       UrhoDrho(t,1)=darhoDrho(t)*du
       UJJ(t,1)=dajj(t)*du
       Unablarho(t,1)=danrnr(t)*du

       Urhorho(t,6)=darhorho(t)*dtu
       Urhotau(t,6)=darhotau(t)*dtu
       UrhoDrho(t,6)=darhoDrho(t)*dtu
       UJJ(t,6)=dajj(t)*dtu
       Unablarho(t,6)=danrnr(t)*dtu

       Urhorho(t,7)=darhorho(t)*dlu
       Urhotau(t,7)=darhotau(t)*dlu
       UrhoDrho(t,7)=darhoDrho(t)*dlu
       UJJ(t,7)=dajj(t)*dlu
       Unablarho(t,7)=danrnr(t)*dlu


       Urhorho(t,3)=sigma*(sigma-1.0_pr)*Cdrho(t)*(rho(0)**sigma)/(rho(0)**2+eps) &
            +darhorho(t)*ddu +ddarhorho(t)*du*du
       Urhotau(t,3)=darhotau(t)*ddu+ddarhotau(t)*du*du
       UrhoDrho(t,3)=darhoDrho(t)*ddu +ddarhoDrho(t)*du*du
       Unablarho(t,3)=danrnr(t)*ddu+ddanrnr(t)*du*du
       UJJ(t,3)=dajj(t)*ddu+ddajj(t)*du*du
    End Do
    Urhorhopr(0,0) = CpV0(0)*(1.0_pr-CpV1(0)*rho(0)/0.16_pr)          &
                    +CpV0(1)*(1.0_pr-CpV1(1)*rho(0)/0.16_pr)
    Urhorhopr(1,0) = CpV0(0)*(1.0_pr-CpV1(0)*rho(0)/0.16_pr)          &
                    +CpV0(1)*(1.0_pr-CpV1(1)*rho(0)/0.16_pr)
    Urhorhopr(2,0) = (CpV0(0)*(1.0_pr-CpV1(0)*rho(0)/0.16_pr)         &
                     -CpV0(1)*(1.0_pr-CpV1(1)*rho(0)/0.16_pr))*2.0_pr
    Urhorhopr(0,1) = (-CpV0(0)*CpV1(0)-CpV0(1)*CpV1(1))/0.16_pr
    Urhorhopr(1,1) = (-CpV0(0)*CpV1(0)-CpV0(1)*CpV1(1))/0.16_pr
    Urhorhopr(2,1) = 2.0_pr*(-CpV0(0)*CpV1(0)+CpV0(1)*CpV1(1))/0.16_pr
    Urhorhopr=Urhorhopr/16.0_pr

    UEnonstdr=0.0_pr; UFnonstdr=0.0_pr; URnonstdr=0.0_pr    
    !
    if (.not.use_j2terms) then
      UJJ=0.0_pr
    end if
    !
  End Subroutine calculate_U_parameters
  !==================================================================================
  !
  !==================================================================================
  Subroutine set_functional_parameters(fname,lpr)
    !--------------------------------------------------------------------------------
    ! RESERVED NAMES ARE:
    !  -namelist forbiden:
    !          'UNRDF'  - best UNEDF
    !          'SKYRME' - best SKYRME
    !  -namelist inforced but not for C-parameters (use_INM=F) 
    !   or NM-parameters (use_INM=T) defined by the solver 
    !          'FITS'   
    !  -namelist inforced (one can overwrite all):
    !          'ANY OTHER NAME' 
    ! i.e., the DME solver defines C-/NM- only using 'FITS'
    !--------------------------------------------------------------------------------
    Implicit None
    Logical, Intent(in) :: lpr
    Character (30), Intent(inout) :: fname 
    Character (30) :: inforcedname     
    Logical :: regularization
    Logical :: use_INM,use_Namelist
    Integer(ipr), Parameter :: lin=15,lout=6,lnamelist=16 
    Real(pr), Dimension(0:1) :: FCrho,FCdrho,FCtau,FCrDr,FCrdJ,FCJ,FCpV0,FCpV1
    Real(pr) :: Fhbzero,Fsigma,Fe2charg
    Real(pr) :: FE_NM,FK_NM,FSMASS_NM,FRHO_NM,FASS_NM,FLASS_NM,FVMASS_NM
    Namelist /UNEDF_NAMELIST/ FunctionalName,DMEorder,DMElda,use_INM,hbzero, &
         Crho,Cdrho,Ctau,CrDr,CrdJ,CJ,sigma,CpV0,CpV1,e2charg, &
         E_NM,K_NM,SMASS_NM,RHO_NM,ASS_NM,LASS_NM,VMASS_NM, &
         mpi,mimu,mivi,miwi,mimi,mibi,mihi, &
         use_cm_cor,use_charge_density,use_j2terms, CExPar, &
         HARTREEselect,Print_Namelist   ,V0trap,hwtrap
    Open(lnamelist,file='MINNESOTA_NAMELIST.DAT',DELIM='APOSTROPHE') ! 'QUOTE' 
    !
    ! parameters
    use_DME3N_terms = .False.
    eps=Spacing(1.0_pr)
    Pi=4.0_pr*Atan(1.0_pr)  
    kfconst=(1.50_pr*Pi**2)**(1.0_pr/3.0_pr)    ! (3Pi^2/2)^(1/3)
    CK=3.0_pr/5.0_pr*kfconst**2
    !
    mimu=0.0_pr; mivi=0.0_pr; miwi=0.0_pr; 
    mimi=0.0_pr; mibi=0.0_pr; mihi=0.0_pr;
    mpi = 0.7_pr;
    HARTREEselect = 0
    CExPar = 1.0_pr
    !
    !==========================
    ! STORE OPTIMIZATION VALUES
    !==========================
    If(Trim(fname).Eq.'FITS') Then
       FCrho=Crho;     FCdrho=Cdrho;     FCtau=Ctau
       FCrDr=CrDr;     FCrdJ=CrdJ;       FCJ=CJ
       FCpV0=CpV0;     FCpV1=CpV1;       
       Fsigma=sigma;   Fhbzero=hbzero
       FE_NM=E_NM;     FK_NM=K_NM;       FSMASS_NM=SMASS_NM; FRHO_NM=RHO_NM
       FASS_NM=ASS_NM; FLASS_NM=LASS_NM; FVMASS_NM=VMASS_NM; 
       Fsigma=sigma;   Fhbzero=hbzero;   Fe2charg=e2charg
    Endif
    Do  

       FUNCTIONAL: Select Case (Trim(fname))
       Case ("MIN1")
          inforcedname='MIN1'
          use_Namelist=.False.
          use_charge_density=.False.
          regularization=.False.
          use_cm_cor=.True. 
          DMEorder=0
          DMElda=0
          mimu=(/ 1.487_pr, 0.639_pr, 0.465_pr /) 
          mivi=(/ 200.0_pr, -178.0_pr, -91.85_pr /) ; 
          miwi=(/ 0.5_pr, 0.25_pr, 0.25_pr /) ; 
          mimi=(/ 0.5_pr, 0.25_pr, 0.25_pr /) ; 
          mibi=(/ 0.0_pr, 0.25_pr, -0.25_pr /) ; 
          mihi=(/ 0.0_pr, 0.25_pr, -0.25_pr /) ;
          hbzero=20.7355300000000007_pr;    
          e2charg = 1.43997840_pr
          Crho = 0.0_pr ; CDrho = 0.0_pr ; Ctau = 0.0_pr
          CrDr = 0.0_pr ; CrdJ = 0.0_pr  ; CJ = 0.0_pr
          use_INM = .false. ; use_J2terms = .true.
          HARTREEselect = 0
          Call CHrho_from_NM()
       Case ("SKYRME")
          inforcedname='SKYRME'
          use_Namelist=.False.
          ! kind of the functional
          use_charge_density=.False.
          regularization=.False.
          use_cm_cor=.True. 
          DMEorder=-1
          DMElda=0
          ! Best SKYRME parameters up to date
          hbzero=20.735530_pr;               sigma=0.1666666666666667_pr    
          Crho(0)=-933.3423749999999472_pr;  Crho(1)=830.0524855000001025_pr
          CDrho(0)=861.0625000000000000_pr;  CDrho(1)=-1064.2732499999999618_pr
          Ctau(0)=57.1286874999999981_pr;    Ctau(1)=24.6567364999999974_pr
          CrDr(0)=-76.9962031249999939_pr;   CrDr(1)=15.6571351249999999_pr 
          CrdJ(0)=-92.2500000000000000_pr;   CrdJ(1)=-30.7500000000000000_pr
          CJ(0)=17.2096115000000012_pr;      CJ(1)=64.5758124999999978_pr
          Cnrho=0.0_pr;                      CJdr=0.0_pr       
          CpV0=-258.20_pr;                   CpV1=0.50_pr
          sigma=0.1666666666666667_pr;     
          hbzero=20.7355300000000007_pr;    
          e2charg = 1.43997840_pr
          ! Associated INM parameters
          E_NM=-15.972149141444596410_pr;   RHO_NM=0.159538756711733398_pr 
          K_NM=229.900964482603626493_pr;   SMASS_NM=1.439546988976078357_pr 
          ASS_NM=32.004302815052007247_pr;  LASS_NM=45.961751480461579433_pr 
          VMASS_NM=1.249838547196253424_pr 
       Case ("FITS")
          inforcedname='FITS'
          use_Namelist=.True.
          use_INM=.False.
       Case default
          inforcedname=fname
          use_Namelist=.True.
          use_INM=.True.
       End Select FUNCTIONAL
       !======================
       ! READ NAMELISTS     
       !======================
       If(.Not.use_Namelist) Exit
       Read(UNIT=lnamelist,NML=UNEDF_NAMELIST,Err=200,End=100)
       If(Trim(FunctionalName).Eq.Trim(inforcedname)) Exit
    Enddo
100 Continue
    Close(lnamelist)
    !
    !================================
    ! SEE WHAT THE NAMELISTS MODIFIED    
    !================================
    INFORCED_FUNCTIONAL: Select Case (Trim(inforcedname))
    Case ("MIN1")
       FunctionalName='MIN1'
    Case ("SKYRME")
       FunctionalName='SKYRME'
    Case ("FITS")
       ! Restore all defined by the DME solver (modify later for use_INM)
       FunctionalName='FITS'
       Crho=FCrho;     Cdrho=FCdrho;     Ctau=FCtau
       CrDr=FCrDr;     CrdJ=FCrdJ;       CJ=FCJ
       CpV0=FCpV0;     CpV1=FCpV1;     
       sigma=Fsigma;   hbzero=Fhbzero;   e2charg=Fe2charg
       E_NM=FE_NM;     K_NM=FK_NM;       SMASS_NM=FSMASS_NM; RHO_NM=FRHO_NM
       ASS_NM=FASS_NM; LASS_NM=FLASS_NM; VMASS_NM=FVMASS_NM; 
    Case default
       ! MISSING ENTRY WITHIN UNEDF_NAMELIST.DAT FILE
       If(Trim(FunctionalName).Ne.Trim(inforcedname)) Then
          Write(LOUT,'(1X,/,A)') 'ATTENTION: MISSING INPUT!'      
          Write(LOUT,*) 'THE INPUT DATA WITH LABEL FUNCTIONALNAME=''',Trim(INFORCEDNAME),''''
          Write(LOUT,*) 'IS MISSING INSIDE THE UNEDF_NAMELIST.DAT FILE.'      
          Write(LOUT,*) 'PLESE CORECT AND TRY AGAIN!'      
          Stop 'PROGRAM STOP IN SET_FUNCTIONAL_PARAMETERS'  
       Endif 
    End Select INFORCED_FUNCTIONAL
    !  
    Call Make_Parameter_Free_Useful_Combintions()
    !
    ! exact Hartree CHrho from INM
    !CHrho=0.0_pr; !!!!If (dmeorder.eq.3) Call CHrho_from_NM()
    !
    If(use_INM) then 
     Call calculate_C_form_NM(E_NM,K_NM,SMASS_NM,RHO_NM,ASS_NM,LASS_NM,VMASS_NM)
    Else
     Crho=Crho+CHrho                       !*(0.0_pr) !mario
    End if
    Call calculate_NM_properties()   
    !
    Crho=Crho-CHrho                        !*(0.00_pr) !mario
    !
    Call calculate_natural_units()
    !
    ! Print output
    If(lpr) Then
     Call print_functional_parameters(lout)
     If(Print_Namelist) Write (lout, UNEDF_NAMELIST)
    Endif 
    !
    Return
    !
200 Continue 
    ! WRONG ENTRY WITHIN UNEDF_NAMELIST.DAT FILE
    Write(LOUT,'(1X,/,A)') 'ATTENTION: WRONG INPUT!'      
    Write(LOUT,*) 'THE INPUT DATA WITH LABEL FUNCTIONALNAME=''',Trim(INFORCEDNAME),''''
    Write(LOUT,*) 'INSIDE THE UNEDF_NAMELIST.DAT FILE IS WRONG.'      
    Write(LOUT,*) 'PLESE CORECT AND TRY AGAIN!'      
    Stop 'PROGRAM STOP IN SET_FUNCTIONAL_PARAMETERS'  
  End Subroutine set_functional_parameters       
  !==================================================================================
  !          
  !==================================================================================
  Subroutine print_functional_parameters(lout)         
    !--------------------------------------------------------------------------------
    Implicit None
    Integer(ipr), Intent(in) :: lout 
    !
    Write(lout,'(a)')   '  ---------------------------------------'
    Write(lout,'(a,a)') '           UNEDF Module Version:', &
                                         Trim(Version)    
    Write(lout,'(a)')   '         M.Kortelainen & M.Stoitsov ' 
    Write(lout,'(a)')   '  ---------------------------------------'
    
    
    Write(lout,'(a)')    
    Write(lout,'(100(2x,a,a,f15.8))') Trim(FunctionalName),' functional'
    Write(lout,'(100(2x,a,f15.8))') '----------------------------------'
    Write(lout,'("  Crho(0)= ",g26.18,"; Crho(1)= ",g26.18)') Crho
    Write(lout,'("  CDrho(0)=",g26.18,"; CDrho(1)=",g26.18)') CDrho
    Write(lout,'("  Ctau(0)= ",g26.18,"; Ctau(1)= ",g26.18)') Ctau
    Write(lout,'("  CrDr(0)= ",g26.18,"; CrDr(1)= ",g26.18)') Crdr
    Write(lout,'("  CrdJ(0)= ",g26.18,"; CrdJ(1)= ",g26.18)') CrdJ
    Write(lout,'("  CJ(0)=   ",g26.18,"; CJ(1)=   ",g26.18)') CJ
    Write(lout,'("  CpV0(0)= ",g26.18,"; CpV0(1)= ",g26.18)') CpV0
    Write(lout,'("  CpV1(0)= ",g26.18,"; CpV1(1)= ",g26.18)') CpV1
    Write(lout,'("  sigma=   ",g26.18,"; hbzero=  ",g26.18)') sigma,hbzero 
    Write(lout,'("  e^2 chrg=",g26.18,"; CExPar=  ",g26.18)') e2charg,CExPar
    Write(lout,'("  c.m. correction: ",L1,", chr. density in direct Coul: ",L1)') use_cm_cor,use_charge_density
    Write(lout,'("  use tensor terms: ",L1)') use_j2terms
        
    Write(lout,'(100(2x,a,f15.8))') 
    Write(lout,'(100(2x,a,f15.8))') 'Coupling constants in natural units'
    Write(lout,'(100(2x,a,f15.8))') '-----------------------------------'
    Write(lout,'("  Crho_nu(0)= ",g26.18,"; Crho_nu(1)= ",g26.18)') nuCrho
    Write(lout,'("  CDrho_nu(0)=",g26.18,"; CDrho_nu(1)=",g26.18)') nuCDrho
    Write(lout,'("  Ctau_nu(0)= ",g26.18,"; Ctau_nu(1)= ",g26.18)') nuCtau
    Write(lout,'("  CrDr_nu(0)= ",g26.18,"; CrDr_nu(1)= ",g26.18)') nuCrdr
    Write(lout,'("  CrdJ_nu(0)= ",g26.18,"; CrdJ_nu(1)= ",g26.18)') nuCrdJ
    Write(lout,'("  CJ_nu(0)=   ",g26.18,"; CJ_nu(1)=   ",g26.18)') nuCJ
    Write(lout,'("  CpV0_nu(0)= ",g26.18,"; CpV0_nu(1)= ",g26.18)') nuCpV0
    Write(lout,'("  CpV1_nu(0)= ",g26.18,"; CpV1_nu(1)= ",g26.18)') nuCpV1
    Write(lout,'("  fpi_nu=     ",g26.18,"; Lambda_nu=  ",g26.18)') nufpi,nuLambda
    
    If (dmeorder.Ge.0) Then
      Write(lout,'(100(2x,a,f15.8))') 
      select case (dmeorder)
      case (0)
        Write(lout,'(100(2x,a,f15.8))',advance='NO') 'Minnesota potential with NV-dme. '
      case (1)
        Write(lout,'(100(2x,a,f15.8))',advance='NO') 'Minnesota potential with PSA-dme. '
      case default
        stop 'wrong dme order'
      end select
     select case (hartreeselect)
      case (-1)
        Write(lout,'(100(2x,a,f15.8))') 'Hartree off'
      case (0)
        Write(lout,'(100(2x,a,f15.8))') 'Exact Hartree'
      case (1)
        Write(lout,'(100(2x,a,f15.8))') 'Hartree from DME'
      case (2)
        Write(lout,'(100(2x,a,f15.8))') 'Hartree from Taylor DME'
      case default
        stop 'wrong Hartree selection'
      end select
      Write(lout,'(100(2x,a,f15.8))') '----------------------------------'
      Write(lout,'("       mu=",3(1x,f12.6))') mimu
      Write(lout,'("       Vi=",3(1x,f12.6))') mivi
      Write(lout,'("       Mi=",3(1x,f12.6))') mimi
      Write(lout,'("       Wi=",3(1x,f12.6))') miwi
      Write(lout,'("       Bi=",3(1x,f12.6))') mibi
      Write(lout,'("       Hi=",3(1x,f12.6))') mihi
      Write(lout,'("  -> CHrho(0) =",f12.6," CHrho(1) =",f12.6)') CHrho
    End If

    Write(lout,'(100(2x,a,f15.8))') 
    Write(lout,'(100(2x,a,f15.8))') ' Harmonic trap parameters'
    Write(lout,'(100(2x,a,f15.8))') '----------------------------------'
    Write(lout,'("     hw = ",f12.6,"   V_0 = ",f12.6)') hwtrap,V0trap
    Write(lout,'(" Vcutoff= ",f12.6)') Vcutoff

    
    Write(lout,'(100(2x,a,f15.8))') 
    Write(lout,'(100(2x,a,f15.8))') 'Nuclear matter properties'
    Write(lout,'(100(2x,a,f15.8))') '----------------------------------'
    Write(lout,'(100(2x,a9,f25.16))') 'E_NM=',E_NM,'K_NM=',K_NM
    Write(lout,'(100(2x,a9,f25.16))') 'P_NM=',P_NM,'RHO_NM=',RHO_NM
    Write(lout,'(100(2x,a9,f25.16))') 'ASS_NM=',ASS_NM,'LASS_NM=',LASS_NM
    Write(lout,'(100(2x,a9,f25.16))') 'SMASS_NM=',SMASS_NM,'VMASS_NM=',VMASS_NM
    
    Call t_from_C()
    Write(lout,'(100(2x,a,f15.8))') 
    Write(lout,'(100(2x,a,f15.8))') 'Associated (t,x)-coupling constants' 
    Write(lout,'(100(2x,a,f15.8))') '-----------------------------------'
    Write(lout,'("  t0=    ",g26.18,"; x0=     ",g26.18)') t0,x0
    Write(lout,'("  t1=    ",g26.18,"; x1=     ",g26.18)') t1,x1
    Write(lout,'("  t2=    ",g26.18,"; x2=     ",g26.18)') t2,x2
    Write(lout,'("  t3=    ",g26.18,"; x3=     ",g26.18)') t3,x3
    Write(lout,'("  b4=    ",g26.18,"; b4p=    ",g26.18)') b4,b4p
    Write(lout,'("  te=    ",g26.18,"; to=     ",g26.18)') te,to
    Write(lout,'("  sigma= ",g26.18,"; hbzero= ",g26.18)') sigma,hbzero
    
    If(Print_Namelist) Then
     Write(lout,'(100(2x,a,f15.8))') 
     SELECTED_FUNCTIONAL: Select Case (Trim(FunctionalName))
      Case ("UNEDF","SKYRME")
       Write(lout,'(100(2x,a,f15.8))') 'NAMELIST CONTENT (cannot be modified for functional names UNEDF,SKYRME)'
       Write(lout,'(100(2x,a,f15.8))') '-----------------------------------------------------------------------'
      Case ("FITS")
       Write(lout,'(100(2x,a,f15.8))') 'NAMELIST CONTENT (Advane usage: modify all but not C-, NM-, and more...)'
       Write(lout,'(100(2x,a,f15.8))') '-----------------------------------------------------------------------'
      Case default
       Write(lout,'(100(2x,a,f15.8))') 'NAMELIST CONTENT (copy/past to UNEDF_NAMELIST.dat and modify)'
       Write(lout,'(100(2x,a,f15.8))') '-------------------------------------------------------------'
     End Select SELECTED_FUNCTIONAL
     Write(lout,'(100(a,f15.8))')    ' !NB: FUNCTIONALNAME should be always in quotations'
    End If
  End Subroutine print_functional_parameters
  !==================================================================================
  !
  !==================================================================================
  Subroutine calculate_natural_units
    ! Calculates coupling constants in natural units
    !-----------------------------------------------------------------------------------
    Implicit None
    nuCrho = Crho*(nufpi**2)/(mevfm**3)
    nuCdrho = Cdrho*(nufpi**2)*((nuLambda*nufpi*nufpi)**sigma)/(mevfm**(3.0_pr*(1.0_pr+sigma)))
    nuCtau = Ctau*((nufpi*nuLambda)**2)/(mevfm**5)
    nuCrDr = CrDr*((nufpi*nuLambda)**2)/(mevfm**5)
    nuCrdJ = CrdJ*((nufpi*nuLambda)**2)/(mevfm**5)
    nuCJ = CJ*((nufpi*nuLambda)**2)/(mevfm**5)
    nuCpV0 = CpV0*(nufpi**2)/(mevfm**3)
    nuCpV1 = CpV1*(nufpi**4)*nuLambda/(mevfm**6)
  End Subroutine calculate_natural_units
  !==================================================================================
  ! 
  !==================================================================================
  Subroutine calculate_C_form_NM(E,K,SMASS,RHO,ASS,LASS,VMASS,sigma_NM)
    ! Calculates volume C-constants (and sigma) form NM properties
    ! Interface usage: 
    !  hbzero,CK,kfconst,mpi,sigma
    !  aRhoRho,bRhoRho...
    !  hRho0Rho0,dhRho0Rho0...
    !  Crho(0),Crho(1),Cdrho(0),Cdrho(1),Ctau(0),Ctau(0)
    !  subroutine calculate_U_parameters
    !
    !  input: E,K,SMASS,RHO,ASS,LASS,VMASS,sigma_NM(optional)
    ! output: Crho(0),Crho(1),Cdrho(0),Cdrho(1),Ctau(0),Ctau(0),sigma(optional)
    ! 
    ! Options:
    !  When sigma_NM exists then 'sigma'=sigma_NM
    !  When sigma_NM does not exist then 'sigma' is defined from NM
    !------------------------------------------------------------------
    Implicit None
    Real(pr), Intent(in) :: E,K,SMASS,RHO,ASS,LASS,VMASS
    Real(pr), Intent(in), Optional :: sigma_NM
    Real(pr) :: aRho0Rho0,daRho0Rho0,ddaRho0Rho0,aRho1Rho1,daRho1Rho1,ddaRho1Rho1
    Real(pr) :: aRho0Tau0,daRho0Tau0,ddaRho0Tau0,aRho1Tau1,daRho1Tau1,ddaRho1Tau1 
    Real(pr) :: u,tauc,rho2
    Real(pr),Parameter :: c13=1.0_pr/3.0_pr,c23=2.0_pr/3.0_pr
    !
    tauc=CK*RHO**c23; u=(kfconst/mpi)*RHO**c13; rho2=rho**2
    Call calculate_U_parameters(RHO,RHO,tauc*RHO,tauc*RHO,0.0_pr,0.0_pr)
    aRho0Rho0=arhorho(0)
    aRho1Rho1=arhorho(1)
    aRho0Tau0=arhotau(0)
    aRho1Tau1=arhotau(1)
    daRho0Rho0=darhorho(0)
    daRho1Rho1=darhorho(1)
    daRho0Tau0=darhotau(0)
    daRho1Tau1=darhotau(1)
    ddaRho0Rho0=ddarhorho(0)
    ddaRho1Rho1=ddarhorho(1)
    ddaRho0Tau0=ddarhotau(0)
    ddaRho1Tau1=ddarhotau(1)
    !
    ! set/calculate sigma
    If (Present(sigma_NM)) Then 
     sigma=sigma_NM 
    Else 
     sigma=((1.0_pr/3.0_pr)*(-K+tauc*hbzero*(-3.0_pr+4.0_pr*SMASS) -9.0_pr*E & 
             +u*RHO*(daRho0Rho0+5.0_pr*tauc*daRho0Tau0 +u*ddaRho0Rho0 & 
             +u*tauc*ddaRho0Tau0)))/(tauc*hbzero*(-3.0_pr+2.0_pr*SMASS)+3.0_pr*E & 
             +u*RHO*(daRho0Rho0+tauc*daRho0Tau0 )) 
    End If
    !  
    Crho(0)=(c13*(tauc*hbzero*(-3.0_pr+(2.0_pr-3.0_pr*sigma)*SMASS) &
        +3.0_pr*(1.0_pr+sigma)*E-3.0_pr*sigma*RHO*aRho0Rho0 &        
        +u*RHO*(daRho0Rho0+tauc*daRho0Tau0)))/(sigma*RHO)
    Cdrho(0)=(c13*RHO**(-1.0_pr-sigma)*(tauc*hbzero*(3.0_pr-2.0_pr*SMASS)&
        -3.0_pr*E -u*RHO*(daRho0Rho0+tauc*daRho0Tau0)))/sigma
    Ctau(0)=(hbzero*(SMASS-1.0_pr)-RHO*(aRho0Tau0))/RHO
    !
    Crho(1)=(27.0_pr*ASS*(1.0_pr+sigma)-9.0_pr*LASS &
        +5.0_pr*tauc*hbzero*(5.0_pr-6.0_pr*VMASS+3.0_pr*sigma*(-4.0_pr+3.0_pr*VMASS)) &
        +20.0_pr*tauc*(2.0_pr-3.0_pr*sigma)*RHO*aRho0Tau0 &
        +RHO*(-27.0_pr*sigma*aRho1Rho1 +40.0_pr*tauc*Ctau(0)-60.0_pr*tauc*sigma*Ctau(0) &
        +5.0_pr*u*tauc*daRho0Tau0+9.0_pr*u*daRho1Rho1+15.0_pr*u*tauc*daRho1Tau1 ))/(27.0_pr*sigma*RHO)
     Cdrho(1)=-(RHO**(-1.0_pr-sigma)*(27.0_pr*ASS-9.0_pr*LASS &
        +5.0_pr*tauc*hbzero*(5.0_pr-6.0_pr*VMASS)+40.0_pr*tauc*RHO*aRho0Tau0 &
        +40.0_pr*tauc*RHO*Ctau(0) +5.0_pr*u*tauc*RHO*daRho0Tau0 &
        +9.0_pr*u*RHO*daRho1Rho1+15.0_pr*u*tauc*RHO*daRho1Tau1))/(27.0_pr*sigma)
     Ctau(1)=(hbzero-hbzero*VMASS+RHO*(aRho0Tau0-aRho1Tau1+Ctau(0)))/RHO
     !
  End Subroutine calculate_C_form_NM
  !==================================================================================
  !
  !==================================================================================
  Subroutine calculate_NM_properties()
    ! Calculates INM properties
    ! Interface usage: 
    !  hbzero,CK,kfconst,mpi,sigma
    !  aRhoRho,bRhoRho...
    !  hRho0Rho0,dhRho0Rho0...
    !  Crho(0),Crho(1),Cdrho(0),Cdrho(1),Ctau(0),Ctau(0)
    !  E_NM,K_NM,SMASS_NM,RHO_NM,ASS_NM,LASS_NM,VMASS_NM,sigma,P_NM,KA_NM
    !  function find_NM_RHOC()
    ! input:  Crho(0),Crho(1),Cdrho(0),Cdrho(1),Ctau(0),Ctau(0),sigma
    ! output: E_NM,K_NM,SMASS_NM,RHO_NM,ASS_NM,LASS_NM,VMASS_NM,sigma,P_NM,KA_NM
    ! Using:
    !  RHO_NM=find_NM_RHOC()
    !------------------------------------------------------------------ 
    Implicit None
    Real(pr) :: aRho0Rho0,daRho0Rho0,ddaRho0Rho0,aRho1Rho1,daRho1Rho1,ddaRho1Rho1
    Real(pr) :: aRho0Tau0,daRho0Tau0,ddaRho0Tau0,aRho1Tau1,daRho1Tau1,ddaRho1Tau1 
    Real(pr) :: u,tauc,rho_NM2
    Real(pr), Parameter :: c13=1.0_pr/3.0_pr,c23=2.0_pr/3.0_pr
    !
    RHO_NM=find_NM_RHOC()
    !
    aRho0Rho0=arhorho(0)
    aRho1Rho1=arhorho(1)
    aRho0Tau0=arhotau(0)
    aRho1Tau1=arhotau(1)
    daRho0Rho0=darhorho(0)
    daRho1Rho1=darhorho(1)
    daRho0Tau0=darhotau(0)
    daRho1Tau1=darhotau(1)
    ddaRho0Rho0=ddarhorho(0)
    ddaRho1Rho1=ddarhorho(1)
    ddaRho0Tau0=ddarhotau(0)
    ddaRho1Tau1=ddarhotau(1)
    tauc=CK*RHO_NM**c23; u=(kfconst/mpi)*RHO_NM**c13; rho_NM2=rho_NM**2
    !
    ! Symmetric Nuclear Matter 
    E_NM=tauc*hbzero+RHO_NM*(aRho0Rho0+Crho(0)+RHO_NM**sigma*Cdrho(0)) &
      +tauc*RHO_NM*(aRho0Tau0+Ctau(0))
    P_NM=c13*RHO_NM**2*((2.0_pr*tauc*hbzero)/RHO_NM+3.0_pr*aRho0Rho0+5.0_pr*tauc*aRho0Tau0 &
      +3.0_pr*Crho(0) &
      +3.0_pr*(1+sigma)*RHO_NM**sigma*Cdrho(0)+5.0_pr*tauc*Ctau(0)+u*daRho0Rho0 &
      +u*tauc*daRho0Tau0)
    SMASS_NM=1.0_pr+(RHO_NM*(aRho0Tau0+Ctau(0)))/hbzero
    K_NM=9.0_pr*sigma*(1+sigma)*RHO_NM**(1+sigma)*Cdrho(0) &
      +(-2.0_pr*tauc*hbzero+10.0_pr*tauc*RHO_NM*aRho0Tau0 &
      +4.0_pr*u*RHO_NM*daRho0Rho0 & 
      +RHO_NM*(10.0_pr*tauc*Ctau(0)+u*(8.0_pr*tauc*daRho0Tau0+u*ddaRho0Rho0 &
      +u*tauc*ddaRho0Tau0)))
    ! 
    ! Asymmetric Nuclear Matter 
    ASS_NM=+RHO_NM*(aRho1Rho1+Crho(1)+RHO_NM**sigma*Cdrho(1)) &
       +(tauc*(5.0_pr*hbzero+RHO_NM*(5.0_pr*aRho0Tau0+15.0_pr*aRho1Tau1 &
       +5.0_pr*(Ctau(0)+3.0_pr*Ctau(1)))))/9.0_pr
    VMASS_NM=(hbzero+RHO_NM*(aRho0Tau0-aRho1Tau1+Ctau(0)-Ctau(1)))/hbzero
    LASS_NM=+3.0_pr*RHO_NM*(aRho1Rho1+Crho(1)+(1.0_pr+sigma)*RHO_NM**sigma*Cdrho(1)) &
       +u*RHO_NM*daRho1Rho1 +(tauc*(10.0_pr*hbzero &
       +25.0_pr*RHO_NM*(aRho0Tau0+3.0_pr*aRho1Tau1+Ctau(0)+3*Ctau(1)) &
       +5.0_pr*u*RHO_NM*(daRho0Tau0+3.0_pr*daRho1Tau1) ))/9.0_pr
    KA_NM=+9.0_pr*sigma*(1.0_pr+sigma)*RHO_NM**(1.0_pr+sigma)*Cdrho(1) &
       +4.0_pr*u*RHO_NM*daRho1Rho1  &
       + u**2*RHO_NM*ddaRho1Rho1 &
       +(tauc*(-10.0_pr*hbzero &
       +50.0_pr*RHO_NM*(aRho0Tau0+3.0_pr*aRho1Tau1+Ctau(0)+3*Ctau(1)) &
       +40.0_pr*u*RHO_NM*(daRho0Tau0+3.0_pr*daRho1Tau1) &
       +5.0_pr*u**2*RHO_NM*(ddaRho0Tau0 &
       +3.0_pr*ddaRho1Tau1)))/9.0_pr
     !       
  End Subroutine calculate_NM_properties
  !==================================================================================  
  !
  !=======================================================================
  Function find_NM_RHOC()
    ! Search for the INM saturation density RHO_NM using the Secant Method
    !====================================================================
    Implicit None
    !Integer(pr) intent(out) :: ierr
    Integer(pr) :: iter
    Real(pr) :: aRho0Rho0,daRho0Rho0,ddaRho0Rho0,aRho1Rho1,daRho1Rho1,ddaRho1Rho1
    Real(pr) :: aRho0Tau0,daRho0Tau0,ddaRho0Tau0,aRho1Tau1,daRho1Tau1,ddaRho1Tau1 
    Real(pr) :: find_NM_RHOC,kfconstmpi,u,tauc
    Real(pr) :: rhom0,rhom,rhom2,w,w0,step,energy
    Real(pr),Parameter :: c13=1.0_pr/3.0_pr,c23=2.0_pr/3.0_pr
    !
    kfconstmpi=kfconst/mpi; step=-0.0010_pr; iter=0
    ! initial point
    rhom=0.170_pr; tauc=CK*rhom**c23; u=kfconstmpi*rhom**c13; rhom2=rhom**2
    Call calculate_U_parameters(rhom,rhom,tauc*rhom,tauc*rhom,0.0_pr,0.0_pr)
    aRho0Rho0=arhorho(0)
    aRho1Rho1=arhorho(1)
    aRho0Tau0=arhotau(0)
    aRho1Tau1=arhotau(1)
    daRho0Rho0=darhorho(0)
    daRho1Rho1=darhorho(1)
    daRho0Tau0=darhotau(0)
    daRho1Tau1=darhotau(1)
    w0=c13*rhom2*((2.0_pr*tauc*hbzero)/rhom+3.0_pr*aRho0Rho0+5.0_pr*tauc*aRho0Tau0 &
      +3.0_pr*Crho(0) &
      +3.0_pr*(1.0_pr+sigma)*rhom**sigma*Cdrho(0)+5.0_pr*tauc*Ctau(0)+u*daRho0Rho0 &
      +u*tauc*daRho0Tau0)
    rhom0=rhom; rhom=rhom+step
    !	
    ! secant method
    Do While(Abs(step).Ge.eps*100.0_pr)
       iter=iter+1
       tauc=CK*rhom**c23; u=kfconstmpi*rhom**c13; rhom2=rhom**2
       Call calculate_U_parameters(rhom,rhom,tauc*rhom,tauc*rhom,0.0_pr,0.0_pr)
       aRho0Rho0=arhorho(0)
       aRho1Rho1=arhorho(1)
       aRho0Tau0=arhotau(0)
       aRho1Tau1=arhotau(1)
       daRho0Rho0=darhorho(0)
       daRho1Rho1=darhorho(1)
       daRho0Tau0=darhotau(0)
       daRho1Tau1=darhotau(1)
       w=c13*rhom2*((2.0_pr*tauc*hbzero)/rhom+3.0_pr*aRho0Rho0+5.0_pr*tauc*aRho0Tau0 &
         +3.0_pr*Crho(0) &
         +3.0_pr*(1.0_pr+sigma)*rhom**sigma*Cdrho(0)+5.0_pr*tauc*Ctau(0)+u*daRho0Rho0 &
         +u*tauc*daRho0Tau0)
       step=-w*(rhom-rhom0)/(w-w0)
       rhom0=rhom; w0=w; rhom=rhom+step       
       If(iter.Gt.100) Stop 'STOP(In find_NM_RHOC)'
       !energy=tauc*hbzero+rhom*(aRho0Rho0+rhom*hRho0Rho0+Crho(0)+rhom**sigma*Cdrho(0)) &
       ! +tauc*rhom*(aRho0Tau0+rhom*hRho0Tau0+Ctau(0))        
       !write(6,'(a,15(1pg12.4))') ' rhom0,rhom,step,e,w=',rhom0,rhom,step,energy,w
    Enddo
    find_NM_RHOC=rhom
  End Function find_NM_RHOC
  !==================================================================================
  !
  !==================================================================================
  Subroutine C_from_t()
    !--------------------------------------------------------------------------------
    ! C- from (t,x)-
    !--------------------------------------------------------------------------------
    Implicit None
     Crho(0) =  3.0_pr/8.0_pr*t0                            
     Cdrho(0) =  (1.0_pr/16.0_pr)*t3
     Crho(1) = -(1.0_pr/4.0_pr)*t0*(0.50_pr+x0)   
     Cdrho(1) = -(1.0_pr/24.0_pr)*t3*(0.50_pr+x3)   
     Ctau(0)  =  (3.0_pr/16.0_pr)*t1+(1.0_pr/4.0_pr)*t2*(5.0_pr/4.0_pr+x2)   
     Ctau(1)  = -(1.0_pr/8.0_pr)*t1*(0.5+x1)+(1.0_pr/8.0_pr)*t2*(0.50_pr+x2)   
     CrDr(0) =  (1.0_pr/16.0_pr)*t2*(5.0_pr/4.0_pr+x2)-(9.0_pr/64.0_pr)*t1
     CrDr(1) =  (3.0_pr/32.0_pr)*t1*(0.5+x1)+(1.0_pr/32.0_pr)*t2*(0.50_pr+x2)   
     CJ(0)    = -(1.0_pr/16.0_pr)*(t1*(2.0_pr*x1-1.0_pr)+t2*(2.0_pr*x2+1)-5*te-15*to)
     CJ(1)    = -(1.0_pr/16.0_pr)*(t2 -t1 + 5.0_pr*te -5.0_pr*to )
     CrdJ(0)   = -b4-(0.50_pr)*b4p              
     CrdJ(1)   = -0.50_pr*b4p                       
  End Subroutine C_from_t
  !==================================================================================
  !
  !==================================================================================
  Subroutine t_from_C()
    !--------------------------------------------------------------------------------
    ! (t,x)- from C-
    !--------------------------------------------------------------------------------
    Implicit None
    t0     =  (8.0_pr/3)*Crho(0)
    t1     =  4.0_pr/3.0_pr*(Ctau(0)-4.0_pr*CrDr(0))         
    t2     =  4.0_pr/3.0_pr*(3.0_pr*Ctau(0)-6.0_pr*Ctau(1)+4.0_pr*CrDr(0)-8.0_pr*CrDr(1))  
    t3     =  16.0_pr*Cdrho(0)
    x0     = -0.50_pr*(3.0_pr*Crho(1)/Crho(0)+1.0_pr)   
    x1     =  2.0_pr*(-Ctau(0)-3.0_pr*Ctau(1)+4.0_pr*CrDr(0)+12.0_pr*CrDr(1))/t1/3.0_pr
    x2     = -2.0_pr*(3.0_pr*Ctau(0)-15.0_pr*Ctau(1)+4.0_pr*CrDr(0)-20.0_pr*CrDr(1))/t2/3.0_pr
    x3     = -0.50_pr*(3.0_pr*Cdrho(1)/Cdrho(0)+1.0_pr) 
    b4     =  CrdJ(1)-CrdJ(0)
    b4p    = -2.0_pr*CrdJ(1)
    te     = (4.0_pr/15.0_pr)*(3.0_pr*CJ(0)-9.0_pr*CJ(1)-4.0_pr*CrDr(0)+12.0_pr*CrDr(1)-2.0_pr*Ctau(0)+6.0_pr*Ctau(1))
    to     = (4.0_pr/15.0_pr)*(3.0_pr*CJ(0)+3.0_pr*CJ(1)+4.0_pr*CrDr(0)+4.0_pr*CrDr(1))
  End Subroutine t_from_C
  !==================================================================================
  !   
  !==================================================================================
  Subroutine CHrho_from_NM()
    !--------------------------------------------------------------------------------
    ! CHrho from NM, E_NM(Hartree)=CHrho*RHO_NM
    !--------------------------------------------------------------------------------
    Implicit None
	Real(pr) :: aux(0:1),auxe
        Integer(ipr) :: i

        aux = 0.0_pr        
        if((dmeorder.ge.0).and.(HARTREEselect.eq.0)) then
          do i=1,3
             auxe = 0.5_pr*(Pi/miMu(i))**1.5_pr
             aux(0) = aux(0) +miVi(i)*(miWi(i)+0.5_pr*(miBi(i)-miHi(i))-0.25_pr*miMi(i))*auxe
             aux(1) = aux(1) -miVi(i)*(0.5_pr*miHi(i)+0.25_pr*miMi(i))*auxe
          end do
        end if
        CHrho = aux
        !
  End Subroutine CHrho_from_NM
  !==================================================================================
  !
  !==================================================================================
  Elemental Function HartreeV00(u)
    !--------------------------------------------------------------------------------
    ! HartreeV(u), E(Hartree)=(1/2)*Int[rho_0(r)*V(|r-r'|)*rho_0(r')]
    !--------------------------------------------------------------------------------
    Implicit None
	Real(pr), Intent(in) :: u 
	Real(pr)             :: HartreeV00
        Real(pr)             :: auxe,auxr
        integer(ipr)         :: i
        !
        auxr = 0.0_pr
        if((dmeorder.ge.0).and.(HARTREEselect.eq.0)) then
          do i=1,3
            auxe = Exp(-miMu(i)*u*u)            
            auxr = auxr +miVi(i)*(miWi(i)+0.5_pr*(miBi(i)-miHi(i))-0.25_pr*miMi(i))*auxe
          end do
        end if
        HartreeV00=auxr
        !
  End Function HartreeV00
  !
  Elemental Function HartreeV11(u)
    !--------------------------------------------------------------------------------
    ! HartreeV(u), E(Hartree)=(1/2)*Int[rho_1(r)*V(|r-r'|)*rho_1(r')]
    !--------------------------------------------------------------------------------
    Implicit None
	Real(pr), Intent(in) :: u 
	Real(pr)             :: HartreeV11
        Real(pr)             :: auxe,auxr
        integer(ipr)         :: i
        !
        auxr = 0.0_pr
        if((dmeorder.ge.0).and.(HARTREEselect.eq.0)) then
          do i=1,3
            auxe = Exp(-miMu(i)*u*u)            
            auxr = auxr -miVi(i)*(0.5_pr*miHi(i)+0.25_pr*miMi(i))*auxe
          end do
        end if
        HartreeV11=auxr
        !
  End Function HartreeV11
  !
  Elemental Function HartreeV01(u)
    !--------------------------------------------------------------------------------
    ! HartreeV(u), E(Hartree)=(1/2)*Int[rho_1(r)*V(|r-r'|)*rho_1(r')]
    !--------------------------------------------------------------------------------
    Implicit None
	Real(pr), Intent(in) :: u 
	Real(pr)             :: HartreeV01
        HartreeV01 = 0.0_pr
  End Function HartreeV01
  !
  !==================================================================================
  !
  !==================================================================================
  Elemental Function ThetaFunction2(u)
    !--------------------------------------------------------------------------------
    ! ThetaFunction2(u)=0 or 1  when x2<2  or x2>2
    !--------------------------------------------------------------------------------
    Implicit None
	Real(pr), Intent(in) :: u 
	Real(pr)             :: x2,ThetaFunction2 
    !
	x2=(u*mpi)
	!
	ThetaFunction2=0.0_pr
	IF(x2.gt.2.0_pr) ThetaFunction2=1.0_pr
	!
  End Function ThetaFunction2
  !================================================================================== 
  !
  !==================================================================================
  Subroutine Make_Parameter_Free_Useful_Combintions()
    !--------------------------------------------------------------------------------
    ! Make Useful combintions
    !--------------------------------------------------------------------------------
    Implicit None
    !
    If (dmeorder.Ge.0) Then
       mpi2=mpi**2
    Endif
    !
  End Subroutine Make_Parameter_Free_Useful_Combintions
  !==================================================================================
  !  
  !==================================================================================
  Elemental Function Vexternal(t,x,y,z)
    !
    Implicit None
    Integer(ipr), Intent(in) :: t
    Real(pr), Intent(in) :: x,y,z
    Real(pr) :: Vexternal
    Real(pr) :: r,aux,vaux
    !
    vaux = 0.5_pr*(hwtrap**2)/(2.0_pr*hbzero)
    r = sqrt(x*x+y*y+z*z)
    aux = V0trap + vaux*r*r
    !
    Vexternal = 0.0_pr
    If (t .eq. 0) then
      Vexternal = aux
    end if
    If (Vexternal .gt. Vcutoff) Vexternal = Vcutoff
    !
  End Function Vexternal
  !
End Module UNEDF
!====================================================================================
!
